import React, { useState, useEffect } from 'react';
import { Button } from './components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from './components/ui/card';
import { Badge } from './components/ui/badge';
import { 
  Shield, 
  TrendingUp, 
  Globe, 
  Menu, 
  X, 
  Bitcoin, 
  ArrowRight,
  ChevronUp,
  ChevronDown,
  Mail,
  Phone,
  MapPin
} from 'lucide-react';
import './App.css';

// Import images
import heroBackground from './assets/hero_background.png';
import aboutUsImage from './assets/about_us.png';
import howToInvestImage from './assets/how_to_invest.png';
import cryptoImage1 from './assets/yG5hEFK68gb5.jpg';
import cryptoImage2 from './assets/hYsg9tvCrccP.jpg';

function App() {
  const [isMenuOpen, setIsMenuOpen] = useState(false);
  const [cryptoPrices, setCryptoPrices] = useState({
    bitcoin: { price: 320450, change: 2.5 },
    ethereum: { price: 18750, change: 1.8 },
    cardano: { price: 2.85, change: -0.5 }
  });

  // Simulate price updates
  useEffect(() => {
    const interval = setInterval(() => {
      setCryptoPrices(prev => ({
        bitcoin: { 
          price: prev.bitcoin.price + (Math.random() - 0.5) * 1000, 
          change: (Math.random() - 0.5) * 10 
        },
        ethereum: { 
          price: prev.ethereum.price + (Math.random() - 0.5) * 100, 
          change: (Math.random() - 0.5) * 8 
        },
        cardano: { 
          price: prev.cardano.price + (Math.random() - 0.5) * 0.1, 
          change: (Math.random() - 0.5) * 6 
        }
      }));
    }, 5000);

    return () => clearInterval(interval);
  }, []);

  const formatPrice = (price) => {
    return new Intl.NumberFormat('pt-BR', {
      style: 'currency',
      currency: 'BRL'
    }).format(price);
  };

  const formatChange = (change) => {
    const sign = change >= 0 ? '+' : '';
    return `${sign}${change.toFixed(2)}%`;
  };

  return (
    <div className="min-h-screen bg-white">
      {/* Navigation */}
      <nav className="fixed top-0 w-full z-50 navbar-blur border-b">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-center h-20">
            <div className="flex items-center">
              <Bitcoin className="h-8 w-8 text-blue-600 mr-2" />
              <span className="text-2xl font-bold text-gradient">CryptoInvest</span>
            </div>
            
            {/* Desktop Menu */}
            <div className="hidden md:flex items-center space-x-8">
              <a href="#home" className="text-gray-700 hover:text-blue-600 transition-colors">Home</a>
              <a href="#cryptos" className="text-gray-700 hover:text-blue-600 transition-colors">Criptomoedas</a>
              <a href="#how-to" className="text-gray-700 hover:text-blue-600 transition-colors">Como Investir</a>
              <a href="#news" className="text-gray-700 hover:text-blue-600 transition-colors">Notícias</a>
              <a href="#about" className="text-gray-700 hover:text-blue-600 transition-colors">Sobre</a>
              <a href="#contact" className="text-gray-700 hover:text-blue-600 transition-colors">Contato</a>
            </div>

            <div className="hidden md:flex items-center space-x-4">
              <Button variant="outline">Entrar</Button>
              <Button className="btn-primary text-white">Começar Agora</Button>
            </div>

            {/* Mobile Menu Button */}
            <div className="md:hidden">
              <Button
                variant="ghost"
                size="icon"
                onClick={() => setIsMenuOpen(!isMenuOpen)}
              >
                {isMenuOpen ? <X className="h-6 w-6" /> : <Menu className="h-6 w-6" />}
              </Button>
            </div>
          </div>
        </div>

        {/* Mobile Menu */}
        {isMenuOpen && (
          <div className="md:hidden bg-white border-t">
            <div className="px-2 pt-2 pb-3 space-y-1">
              <a href="#home" className="block px-3 py-2 text-gray-700">Home</a>
              <a href="#cryptos" className="block px-3 py-2 text-gray-700">Criptomoedas</a>
              <a href="#how-to" className="block px-3 py-2 text-gray-700">Como Investir</a>
              <a href="#news" className="block px-3 py-2 text-gray-700">Notícias</a>
              <a href="#about" className="block px-3 py-2 text-gray-700">Sobre</a>
              <a href="#contact" className="block px-3 py-2 text-gray-700">Contato</a>
              <div className="px-3 py-2 space-y-2">
                <Button variant="outline" className="w-full">Entrar</Button>
                <Button className="w-full btn-primary text-white">Começar Agora</Button>
              </div>
            </div>
          </div>
        )}
      </nav>

      {/* Hero Section */}
      <section id="home" className="pt-20 hero-gradient text-white relative overflow-hidden">
        <div 
          className="absolute inset-0 opacity-20"
          style={{
            backgroundImage: `url(${heroBackground})`,
            backgroundSize: 'cover',
            backgroundPosition: 'center'
          }}
        />
        <div className="relative max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-24">
          <div className="text-center">
            <h1 className="text-5xl md:text-6xl font-bold mb-6 animate-fade-in">
              Invista no Futuro com Criptomoedas
            </h1>
            <p className="text-xl md:text-2xl mb-8 max-w-3xl mx-auto opacity-90">
              Plataforma segura e intuitiva para seus investimentos em moedas digitais
            </p>
            <p className="text-lg mb-10 max-w-2xl mx-auto opacity-80">
              Descubra o potencial das criptomoedas com nossa plataforma confiável. 
              Invista com segurança e acompanhe seus resultados em tempo real.
            </p>
            <div className="flex flex-col sm:flex-row gap-4 justify-center">
              <Button size="lg" className="bg-white text-blue-600 hover:bg-gray-100 text-lg px-8 py-4">
                Comece a Investir
                <ArrowRight className="ml-2 h-5 w-5" />
              </Button>
              <Button size="lg" variant="outline" className="border-white text-white hover:bg-white hover:text-blue-600 text-lg px-8 py-4">
                Saiba Mais
              </Button>
            </div>
          </div>
        </div>
      </section>

      {/* Benefits Section */}
      <section className="section-padding bg-gray-50">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-16">
            <h2 className="text-4xl font-bold text-gray-900 mb-4">
              Por que Escolher Criptomoedas?
            </h2>
            <p className="text-xl text-gray-600 max-w-2xl mx-auto">
              Descubra as vantagens que fazem das criptomoedas uma excelente opção de investimento
            </p>
          </div>
          
          <div className="grid md:grid-cols-3 gap-8">
            <Card className="crypto-card text-center p-8">
              <CardContent className="pt-6">
                <div className="crypto-icon bg-blue-100 text-blue-600">
                  <Shield className="h-6 w-6" />
                </div>
                <h3 className="text-xl font-semibold mb-4">Segurança Blockchain</h3>
                <p className="text-gray-600">
                  Tecnologia blockchain garante transações seguras e transparentes
                </p>
              </CardContent>
            </Card>

            <Card className="crypto-card text-center p-8">
              <CardContent className="pt-6">
                <div className="crypto-icon bg-green-100 text-green-600">
                  <TrendingUp className="h-6 w-6" />
                </div>
                <h3 className="text-xl font-semibold mb-4">Potencial de Crescimento</h3>
                <p className="text-gray-600">
                  Oportunidades de valorização superiores aos investimentos tradicionais
                </p>
              </CardContent>
            </Card>

            <Card className="crypto-card text-center p-8">
              <CardContent className="pt-6">
                <div className="crypto-icon bg-purple-100 text-purple-600">
                  <Globe className="h-6 w-6" />
                </div>
                <h3 className="text-xl font-semibold mb-4">Acessibilidade Global</h3>
                <p className="text-gray-600">
                  Invista 24/7 de qualquer lugar do mundo com facilidade
                </p>
              </CardContent>
            </Card>
          </div>
        </div>
      </section>

      {/* Popular Cryptocurrencies */}
      <section id="cryptos" className="section-padding">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-16">
            <h2 className="text-4xl font-bold text-gray-900 mb-4">
              Principais Criptomoedas
            </h2>
            <p className="text-xl text-gray-600">
              Acompanhe as moedas digitais mais negociadas
            </p>
          </div>

          <div className="grid md:grid-cols-3 gap-8">
            {/* Bitcoin */}
            <Card className="crypto-card">
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Bitcoin</CardTitle>
                <div className="bitcoin-icon w-8 h-8 rounded-full flex items-center justify-center">
                  <span className="text-white font-bold text-xs">₿</span>
                </div>
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">{formatPrice(cryptoPrices.bitcoin.price)}</div>
                <div className={`flex items-center text-sm ${cryptoPrices.bitcoin.change >= 0 ? 'price-positive' : 'price-negative'}`}>
                  {cryptoPrices.bitcoin.change >= 0 ? <ChevronUp className="h-4 w-4" /> : <ChevronDown className="h-4 w-4" />}
                  {formatChange(cryptoPrices.bitcoin.change)}
                </div>
                <p className="text-xs text-gray-500 mt-2">BTC</p>
              </CardContent>
            </Card>

            {/* Ethereum */}
            <Card className="crypto-card">
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Ethereum</CardTitle>
                <div className="ethereum-icon w-8 h-8 rounded-full flex items-center justify-center">
                  <span className="text-white font-bold text-xs">Ξ</span>
                </div>
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">{formatPrice(cryptoPrices.ethereum.price)}</div>
                <div className={`flex items-center text-sm ${cryptoPrices.ethereum.change >= 0 ? 'price-positive' : 'price-negative'}`}>
                  {cryptoPrices.ethereum.change >= 0 ? <ChevronUp className="h-4 w-4" /> : <ChevronDown className="h-4 w-4" />}
                  {formatChange(cryptoPrices.ethereum.change)}
                </div>
                <p className="text-xs text-gray-500 mt-2">ETH</p>
              </CardContent>
            </Card>

            {/* Cardano */}
            <Card className="crypto-card">
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Cardano</CardTitle>
                <div className="cardano-icon w-8 h-8 rounded-full flex items-center justify-center">
                  <span className="text-white font-bold text-xs">₳</span>
                </div>
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">{formatPrice(cryptoPrices.cardano.price)}</div>
                <div className={`flex items-center text-sm ${cryptoPrices.cardano.change >= 0 ? 'price-positive' : 'price-negative'}`}>
                  {cryptoPrices.cardano.change >= 0 ? <ChevronUp className="h-4 w-4" /> : <ChevronDown className="h-4 w-4" />}
                  {formatChange(cryptoPrices.cardano.change)}
                </div>
                <p className="text-xs text-gray-500 mt-2">ADA</p>
              </CardContent>
            </Card>
          </div>

          <div className="text-center mt-12">
            <Button size="lg" variant="outline" className="text-blue-600 border-blue-600 hover:bg-blue-50">
              Ver Todas as Criptomoedas
            </Button>
          </div>
        </div>
      </section>

      {/* How to Invest */}
      <section id="how-to" className="section-padding bg-gray-50">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="grid lg:grid-cols-2 gap-12 items-center">
            <div>
              <h2 className="text-4xl font-bold text-gray-900 mb-6">
                Como Começar a Investir
              </h2>
              <p className="text-xl text-gray-600 mb-8">
                Siga estes passos simples para começar sua jornada no mundo das criptomoedas
              </p>
              
              <div className="space-y-6">
                <div className="flex items-start">
                  <div className="flex-shrink-0 w-8 h-8 bg-blue-600 text-white rounded-full flex items-center justify-center font-bold mr-4">
                    1
                  </div>
                  <div>
                    <h3 className="text-lg font-semibold mb-2">Cadastre-se</h3>
                    <p className="text-gray-600">Crie sua conta em menos de 2 minutos</p>
                  </div>
                </div>
                
                <div className="flex items-start">
                  <div className="flex-shrink-0 w-8 h-8 bg-blue-600 text-white rounded-full flex items-center justify-center font-bold mr-4">
                    2
                  </div>
                  <div>
                    <h3 className="text-lg font-semibold mb-2">Deposite</h3>
                    <p className="text-gray-600">Transfira fundos de forma segura</p>
                  </div>
                </div>
                
                <div className="flex items-start">
                  <div className="flex-shrink-0 w-8 h-8 bg-blue-600 text-white rounded-full flex items-center justify-center font-bold mr-4">
                    3
                  </div>
                  <div>
                    <h3 className="text-lg font-semibold mb-2">Invista</h3>
                    <p className="text-gray-600">Escolha suas criptomoedas e comece</p>
                  </div>
                </div>
              </div>

              <Button size="lg" className="btn-primary text-white mt-8">
                Começar Agora
                <ArrowRight className="ml-2 h-5 w-5" />
              </Button>
            </div>
            
            <div className="lg:order-first">
              <img 
                src={howToInvestImage} 
                alt="Como investir em criptomoedas" 
                className="w-full h-auto rounded-lg shadow-lg"
              />
            </div>
          </div>
        </div>
      </section>

      {/* News Section */}
      <section id="news" className="section-padding">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-16">
            <h2 className="text-4xl font-bold text-gray-900 mb-4">
              Últimas Notícias
            </h2>
            <p className="text-xl text-gray-600">
              Mantenha-se atualizado com o mercado de criptomoedas
            </p>
          </div>

          <div className="grid md:grid-cols-3 gap-8">
            <Card className="crypto-card">
              <CardContent className="p-6">
                <Badge className="mb-4">Mercado</Badge>
                <h3 className="text-lg font-semibold mb-2">
                  Bitcoin atinge nova máxima histórica
                </h3>
                <p className="text-gray-600 text-sm mb-4">
                  A principal criptomoeda do mundo alcançou um novo recorde, superando todas as expectativas dos analistas...
                </p>
                <p className="text-xs text-gray-500">Há 2 horas</p>
              </CardContent>
            </Card>

            <Card className="crypto-card">
              <CardContent className="p-6">
                <Badge className="mb-4">Tecnologia</Badge>
                <h3 className="text-lg font-semibold mb-2">
                  Ethereum 2.0: O que esperar
                </h3>
                <p className="text-gray-600 text-sm mb-4">
                  A atualização mais aguardada do Ethereum promete revolucionar a rede com maior eficiência...
                </p>
                <p className="text-xs text-gray-500">Há 5 horas</p>
              </CardContent>
            </Card>

            <Card className="crypto-card">
              <CardContent className="p-6">
                <Badge className="mb-4">Regulamentação</Badge>
                <h3 className="text-lg font-semibold mb-2">
                  Regulamentação de cripto no Brasil
                </h3>
                <p className="text-gray-600 text-sm mb-4">
                  Novas diretrizes do Banco Central trazem mais segurança para investidores brasileiros...
                </p>
                <p className="text-xs text-gray-500">Há 1 dia</p>
              </CardContent>
            </Card>
          </div>
        </div>
      </section>

      {/* About Section */}
      <section id="about" className="section-padding bg-gray-50">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="grid lg:grid-cols-2 gap-12 items-center">
            <div>
              <img 
                src={aboutUsImage} 
                alt="Nossa equipe" 
                className="w-full h-auto rounded-lg shadow-lg"
              />
            </div>
            
            <div>
              <h2 className="text-4xl font-bold text-gray-900 mb-6">
                Sobre a CryptoInvest
              </h2>
              <p className="text-xl text-gray-600 mb-6">
                Nossa missão é democratizar o acesso aos investimentos em criptomoedas
              </p>
              <p className="text-gray-600 mb-6">
                Somos uma equipe de especialistas em tecnologia financeira e blockchain, 
                comprometidos em oferecer a melhor experiência de investimento em criptomoedas 
                para nossos usuários.
              </p>
              <p className="text-gray-600 mb-8">
                Com anos de experiência no mercado financeiro e tecnológico, desenvolvemos 
                uma plataforma segura, intuitiva e confiável para que você possa investir 
                com tranquilidade.
              </p>
              <Button size="lg" variant="outline" className="text-blue-600 border-blue-600 hover:bg-blue-50">
                Conheça Nossa Equipe
              </Button>
            </div>
          </div>
        </div>
      </section>

      {/* CTA Section */}
      <section className="section-padding hero-gradient text-white">
        <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
          <h2 className="text-4xl font-bold mb-6">
            Pronto para Começar?
          </h2>
          <p className="text-xl mb-8 opacity-90">
            Junte-se a milhares de investidores que já descobriram o potencial das criptomoedas
          </p>
          <Button size="lg" className="bg-white text-blue-600 hover:bg-gray-100 text-lg px-8 py-4">
            Criar Conta Grátis
            <ArrowRight className="ml-2 h-5 w-5" />
          </Button>
        </div>
      </section>

      {/* Contact Section */}
      <section id="contact" className="section-padding">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-16">
            <h2 className="text-4xl font-bold text-gray-900 mb-4">
              Entre em Contato
            </h2>
            <p className="text-xl text-gray-600">
              Estamos aqui para ajudar você a começar sua jornada
            </p>
          </div>

          <div className="grid md:grid-cols-3 gap-8">
            <Card className="text-center p-8">
              <CardContent className="pt-6">
                <Mail className="h-12 w-12 text-blue-600 mx-auto mb-4" />
                <h3 className="text-lg font-semibold mb-2">Email</h3>
                <p className="text-gray-600">contato@cryptoinvest.com</p>
              </CardContent>
            </Card>

            <Card className="text-center p-8">
              <CardContent className="pt-6">
                <Phone className="h-12 w-12 text-blue-600 mx-auto mb-4" />
                <h3 className="text-lg font-semibold mb-2">Telefone</h3>
                <p className="text-gray-600">(11) 9999-9999</p>
              </CardContent>
            </Card>

            <Card className="text-center p-8">
              <CardContent className="pt-6">
                <MapPin className="h-12 w-12 text-blue-600 mx-auto mb-4" />
                <h3 className="text-lg font-semibold mb-2">Endereço</h3>
                <p className="text-gray-600">São Paulo, SP</p>
              </CardContent>
            </Card>
          </div>
        </div>
      </section>

      {/* Footer */}
      <footer className="bg-gray-900 text-white py-12">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="grid md:grid-cols-4 gap-8">
            <div>
              <div className="flex items-center mb-4">
                <Bitcoin className="h-8 w-8 text-blue-400 mr-2" />
                <span className="text-xl font-bold">CryptoInvest</span>
              </div>
              <p className="text-gray-400">
                Sua plataforma confiável para investimentos em criptomoedas.
              </p>
            </div>

            <div>
              <h3 className="text-lg font-semibold mb-4">Links Úteis</h3>
              <ul className="space-y-2 text-gray-400">
                <li><a href="#" className="hover:text-white transition-colors">FAQ</a></li>
                <li><a href="#" className="hover:text-white transition-colors">Suporte</a></li>
                <li><a href="#" className="hover:text-white transition-colors">Termos de Uso</a></li>
                <li><a href="#" className="hover:text-white transition-colors">Política de Privacidade</a></li>
              </ul>
            </div>

            <div>
              <h3 className="text-lg font-semibold mb-4">Criptomoedas</h3>
              <ul className="space-y-2 text-gray-400">
                <li><a href="#" className="hover:text-white transition-colors">Bitcoin</a></li>
                <li><a href="#" className="hover:text-white transition-colors">Ethereum</a></li>
                <li><a href="#" className="hover:text-white transition-colors">Cardano</a></li>
                <li><a href="#" className="hover:text-white transition-colors">Todas as Moedas</a></li>
              </ul>
            </div>

            <div>
              <h3 className="text-lg font-semibold mb-4">Contato</h3>
              <ul className="space-y-2 text-gray-400">
                <li>contato@cryptoinvest.com</li>
                <li>(11) 9999-9999</li>
                <li>São Paulo, SP</li>
              </ul>
            </div>
          </div>

          <div className="border-t border-gray-800 mt-8 pt-8 text-center text-gray-400">
            <p>&copy; 2025 CryptoInvest. Todos os direitos reservados.</p>
          </div>
        </div>
      </footer>
    </div>
  );
}

export default App;
